-- File: scripts/TakeObjectMain.lua
--[[ 
Mod: Take Single Objects

Author: Lism
Date: 2025-09-13
Version: 1.0.0.0

Changelog:
    v 1.0.0.0 @2025-09-13 - Initial release
--]]

TakeObjectMain = TakeObjectMain or {}
TakeSingleObjects = TakeSingleObjects or { registry = {}, version = "1.0.0.0", API_VERSION = 2, isActive = true }

-- Basisparameter
local MAX_USE_DISTANCE   = 2.0
local MIN_FREE_HEIGHT    = 0.12
local DEFAULT_OBJ_LITERS = 25
local ADD_RADIUS         = 2.25
local EPS                = 0.001
local DBG                = false

local COLLISION_MASK = CollisionFlag.DEFAULT + CollisionFlag.STATIC_OBJECT + CollisionFlag.DYNAMIC_OBJECT + CollisionFlag.VEHICLE

-- Utils
local function dlog(fmt, ...) if DBG then Logging.info("[TakeSingleObjects] " .. string.format(fmt, ...)) end end
local function wlog(fmt, ...) Logging.warning("[TakeSingleObjects] " .. string.format(fmt, ...)) end
local function normPath(p) if p == nil then return "" end p = string.gsub(p, "\\", "/"); return string.lower(p) end
local function endsWith(str, suffix) if not str or not suffix then return false end local ls,le=#str,#suffix if le>ls then return false end return string.sub(str, ls-le+1, ls)==suffix end
local function basename(p) if not p or p=="" then return p end local j=1 local k=0 while true do k = string.find(p, "/", j, true); if not k then break end j = k+1 end return string.sub(p, j) end
local function objectConfigPath(obj) local cfg = obj and (obj.configFileName or obj.configFile) or ""; return normPath(cfg) end

-- FillTypes
local function resolveFillTypeIndex(ft)
    if ft == nil then return nil end
    if type(ft) == "number" then return ft end
    if g_fillTypeManager ~= nil then
        return g_fillTypeManager:getFillTypeIndexByName(ft)
            or g_fillTypeManager:getFillTypeIndexByName(string.lower(ft))
            or g_fillTypeManager:getFillTypeIndexByName(string.upper(ft))
    end
    return nil
end

-- Registry/Mapping (API v2)
local function normalizeObjectXmlPath(objRel, modDir)
    if objRel == nil or objRel == "" then return nil end
    if string.sub(objRel, 1, 8):lower() == "$moddir$" then
        local rest = string.sub(objRel, 9)
        if string.sub(rest, 1, 1) == "/" then rest = string.sub(rest, 2) end
        return Utils.getFilename(rest, modDir)
    end
    if string.sub(objRel, 1, 1) == "$" or string.sub(objRel, 1, 1) == "/" then return objRel end
    return Utils.getFilename(objRel, modDir)
end
local function normalizeEndsWithPath(endsRel, modDir)
    if endsRel == nil or endsRel == "" then return "" end
    local s = endsRel
    if string.sub(s, 1, 8):lower() == "$moddir$" then
        local rest = string.sub(s, 9)
        if string.sub(rest, 1, 1) == "/" then rest = string.sub(rest, 2) end
        s = Utils.getFilename(rest, modDir)
    end
    return normPath(s)
end

local function register(def)
    if def == nil or def.match == nil or def.match.endsWith == nil or def.object == nil or def.object.xml == nil then return false end
    def.__ends = normPath(def.match.endsWith)
    def.id = def.id or def.__ends
    def.object.liters = def.object.liters or def.object.amount or DEFAULT_OBJ_LITERS
    def.object.fillTypeIndex = def.object.fillTypeIndex or resolveFillTypeIndex(def.object.fillType)
    def.object.isConsumable = def.object.isConsumable == true
    def.object.configurable = def.object.configurable == true
    def.object.allowRemainder = def.object.allowRemainder ~= false -- default true
    TakeSingleObjects.registry[def.id] = def
    return true
end

local function findMappingXmlPath(dir)
    local p1 = Utils.getFilename("tso_mappings.xml", dir)
    if fileExists(p1) then return p1 end
    local p2 = Utils.getFilename("tso_mapping.xml", dir)
    if fileExists(p2) then return p2 end
    return nil
end

local function loadMappingsFromMod(mod)
    local dir = mod and (mod.modDir or mod.modFile or mod.modDirOverride) or nil
    if type(dir) ~= "string" or dir == "" then return 0 end
    local xmlPath = findMappingXmlPath(dir)
    if xmlPath == nil then return 0 end

    local xml = XMLFile.load("tsoMappings", xmlPath, "TakeSingleObjectsMappings")
    if xml == nil then return 0 end

    local apiV = xml:getInt("TakeSingleObjectsMappings#apiVersion", 1)
    local cnt = 0
    local i = 0
    while true do
        local key = string.format("TakeSingleObjectsMappings.mapping(%d)", i)
        if not xml:hasProperty(key) then break end
        local endsRaw = xml:getString(key .. "#endsWith") or xml:getString(key .. "#endswith")
        local ends = normalizeEndsWithPath(endsRaw, dir)
        local objRel = xml:getString(key .. "#objXml") or xml:getString(key .. "#objxml")
        local liters = xml:getInt(key .. "#amount", xml:getInt(key .. "#liters", DEFAULT_OBJ_LITERS))
        local ftName = xml:getString(key .. "#fillType") or xml:getString(key .. "#filltype")
        local allow  = xml:getBool(key .. "#allowRemainder", true)
        local isCons = xml:getBool(key .. "#isConsumable", xml:getBool(key .. "#consumable", false))
        local cfgbl  = xml:getBool(key .. "#configurable", false)
        local idRaw = xml:getString(key .. "#id")
        local modId = (mod and (mod.name or mod.modName)) or "tso"
        local id
        if idRaw ~= nil and idRaw ~= "" then
            id = string.format("%s.%s", modId, idRaw)
        else
            -- Fallback <modname in lowercase>.<index>
            id = string.format("%s.%d", string.lower(modId), i)
        end

        if ends ~= "" and objRel ~= nil then
            local objXmlFull = normalizeObjectXmlPath(objRel, dir)
            local def = {
                id = id,
                apiVersion = apiV,
                match = { endsWith = ends },
                object = {
                    xml = objXmlFull,
                    liters = liters,
                    amount = liters,
                    fillType = ftName,
                    fillTypeIndex = resolveFillTypeIndex(ftName),
                    allowRemainder = allow,
                    isConsumable = isCons,
                    configurable = cfgbl
                }
            }
            register(def)
            cnt = cnt + 1
        end
        i = i + 1
    end
    xml:delete()
    dlog("Loaded %d mappings from %s (api=%s)", cnt, basename(xmlPath), tostring(apiV))
    return cnt
end

local function loadAllMappings()
    if g_modManager == nil then return end
    TakeSingleObjects.registry = {}
    local mods = g_modManager.mods or (g_modManager.getMods and g_modManager:getMods()) or {}
    local sum = 0
    for _, m in pairs(mods) do sum = sum + (loadMappingsFromMod(m) or 0) end
    if sum == 0 then wlog("Keine tso_mappings.xml gefunden – Mod kann ohne Mappings nicht arbeiten (API v2).") end
end

local function ensureMappingsLoaded()
    if not TakeSingleObjects._mappingsLoaded and g_modManager ~= nil then loadAllMappings(); TakeSingleObjects._mappingsLoaded = true end
end

-- FillUnit helpers
local function pickFillUnitFor(obj, wantedFillTypeIndex)
    local spec = obj and (obj.spec_fillUnit or obj.spec_fillunit)
    if spec ~= nil and spec.fillUnits ~= nil then
        for i=1, #spec.fillUnits do
            local fu = spec.fillUnits[i]
            local lvl = fu and fu.fillLevel or 0
            local ft  = (fu and (fu.fillType or fu.lastValidFillType)) or nil
            if lvl > 0 then
                if wantedFillTypeIndex == nil or ft == wantedFillTypeIndex or ft == nil then
                    return i, lvl, ft or wantedFillTypeIndex
                end
            end
        end
        for i=1, #spec.fillUnits do
            local fu = spec.fillUnits[i]
            if wantedFillTypeIndex == nil or (fu and fu.supportedFillTypes and fu.supportedFillTypes[wantedFillTypeIndex]) then
                local ft = (fu and (fu.fillType or fu.lastValidFillType)) or wantedFillTypeIndex
                return i, fu.fillLevel or 0, ft
            end
        end
    end
    return nil, 0, nil
end

local function getUnitTextFor(obj, fuIndex)
    local spec = obj and (obj.spec_fillUnit or obj.spec_fillunit)
    if spec and spec.fillUnits and spec.fillUnits[fuIndex] then
        local fu = spec.fillUnits[fuIndex]
        if fu.unitText ~= nil and fu.unitText ~= "" then return fu.unitText end
    end
    return "L"
end

local function computeFirstObjectLiters(level, objSize, allowRemainder)
    local lvlI = math.floor((level or 0) + 0.5)
    if lvlI <= 0 then return 0 end
    objSize = objSize or DEFAULT_OBJ_LITERS
    if allowRemainder then local r = lvlI % objSize; if r > 0 then return r end; if lvlI >= objSize then return objSize end; return 0
    else if lvlI >= objSize then return objSize end; return 0 end
end

-- Mapping finden
local function findDefForObject(obj)
    local cfg = objectConfigPath(obj)
    for _, d in pairs(TakeSingleObjects.registry) do
        if d.__ends ~= "" and endsWith(cfg, d.__ends) then return d end
    end
    return nil
end

-- Raycast
local function computeRay(cam)
    local cx,cy,cz = getWorldTranslation(cam)
    local tx,ty,tz = unProject(0.5, 0.5, 1)
    local dx,dy,dz = MathUtil.vector3Normalize(tx - cx, ty - cy, tz - cz)
    return cx,cy,cz, dx,dy,dz
end

function TakeObjectMain:onRaycastHit(hitNode, x, y, z, distance, nx, ny, nz, subShapeId)
    if hitNode == nil or hitNode == 0 or distance > MAX_USE_DISTANCE then return true end

    local obj; if g_currentMission ~= nil then obj = g_currentMission:getNodeObject(hitNode) end
    if obj == nil then local node = hitNode; for _=1,20 do if node == nil or node == 0 then break end if g_currentMission ~= nil then obj = g_currentMission:getNodeObject(node) end if obj ~= nil then break end node = getParent(node) end end
    if obj == nil then return true end

    local fu, lvl, ft = pickFillUnitFor(obj, nil)
    if fu == nil then return true end

    local def = findDefForObject(obj)
    if def == nil then return true end
    if def.object.fillTypeIndex ~= nil and ft ~= nil and ft ~= def.object.fillTypeIndex then return true end

    local wish = def.object.liters or DEFAULT_OBJ_LITERS
    local takeLiters = computeFirstObjectLiters(lvl, wish, def.object.allowRemainder == true)

    self.currentTarget = { object=obj, def=def, fu=fu, lvl=lvl, ft=ft, x=x, y=(y or 0) + MIN_FREE_HEIGHT, z=z, nx=nx, ny=ny, nz=nz }

    if (takeLiters or 0) > 0 and TakeObjectMain.actionEventIdTake and g_inputBinding then
        local unitText = getUnitTextFor(obj, fu)
        local label = string.format("%s [%d %s]", g_i18n:getText("input_TAKE_OBJECT"), takeLiters, unitText)
        g_inputBinding:setActionEventText(TakeObjectMain.actionEventIdTake, label)
        if g_inputBinding.setActionEventTextVisibility then g_inputBinding:setActionEventTextVisibility(TakeObjectMain.actionEventIdTake, true) end
    end
    return false
end

-- Nearby-Object Scan
function TakeObjectMain:_overlapAddFn(node)
    if node == nil or node == 0 then return true end
    if self._overlap == nil then return true end
    local obj = g_currentMission and g_currentMission:getNodeObject(node) or nil
    if obj == nil or obj.isDeleted then return true end
    if self._overlap.seenObj[obj] then return true end
    self._overlap.seenObj[obj] = true
    table.insert(self._overlap.found, obj)
    return true
end

local function xmlMatches(cfg, want)
    if want == nil or want == "" then return false end
    if cfg == nil or cfg == "" then return false end
    if endsWith(cfg, want) then return true end
    local bn = basename(want)
    return (bn and bn ~= "" and endsWith(cfg, "/"..bn)) or false
end

function TakeObjectMain:_findNearbyObjects(cx, cy, cz, def, farmId)
    local wantXml = normPath(def and def.object and def.object.xml)
    local ft = def and def.object and def.object.fillTypeIndex

    self._overlap = { found = {}, seenObj = {} }
    overlapSphere(cx, cy, cz, ADD_RADIUS, "_overlapAddFn", self, COLLISION_MASK, true, true, true)
    local candidates = self._overlap.found or {}
    self._overlap = nil

    local list, total = {}, 0
    for _, v in ipairs(candidates) do
        repeat
            if v.getOwnerFarmId ~= nil and farmId ~= nil then local vf = v:getOwnerFarmId() or 0; if vf ~= 0 and vf ~= farmId then break end end
            if not xmlMatches(objectConfigPath(v), wantXml) then break end
            if ft ~= nil and v.getFillUnitSupportsFillType ~= nil then
                local ok = false
                for _, fu in ipairs(v.getFillUnits and v:getFillUnits() or {}) do
                    local idx = fu.fillUnitIndex or fu.index or 1
                    if v:getFillUnitSupportsFillType(idx, ft) then ok = true break end
                end
                if not ok then break end
            end
            local liters = 0
            for _, fu in ipairs(v.getFillUnits and v:getFillUnits() or {}) do liters = liters + (fu.fillLevel or 0) end
            if liters <= EPS then break end
            local nid = NetworkUtil.getObjectId(v)
            table.insert(list, { v=v, id=nid, liters=liters })
            total = total + liters
        until true
    end
    return list, total
end

-- Update / Actions
TakeObjectMain.actionEventIdTake = nil
TakeObjectMain.actionEventIdAdd  = nil
TakeObjectMain.currentTarget     = nil
TakeObjectMain.addContext        = nil

function TakeObjectMain:update(dt)
    ensureMappingsLoaded()

    local player = g_localPlayer or (g_currentMission and g_currentMission.player); if player == nil then return end
    local cam = (player.getCurrentCameraNode and player:getCurrentCameraNode()) or player.cameraNode; if cam == nil then return end

    local x,y,z, dx,dy,dz = computeRay(cam)
    self.currentTarget = nil
    self.addContext = nil

    if TakeObjectMain.actionEventIdTake and g_inputBinding and g_inputBinding.setActionEventTextVisibility then g_inputBinding:setActionEventTextVisibility(TakeObjectMain.actionEventIdTake, false) end
    if TakeObjectMain.actionEventIdAdd  and g_inputBinding and g_inputBinding.setActionEventTextVisibility  then g_inputBinding:setActionEventTextVisibility(TakeObjectMain.actionEventIdAdd,  false) end

    raycastClosest(x,y,z, dx,dy,dz, MAX_USE_DISTANCE, "onRaycastHit", TakeObjectMain, COLLISION_MASK, true, player.rootNode, cam)

    if self.currentTarget ~= nil then
        local t = self.currentTarget
        local farmIdOwner = (t.object and t.object.getOwnerFarmId and t.object:getOwnerFarmId()) or 0
        if farmIdOwner == 0 then
            farmIdOwner = (g_localPlayer and g_localPlayer.farmId) or (g_currentMission and g_currentMission.getFarmId and g_currentMission:getFarmId()) or 1
        end
        local list, total = self:_findNearbyObjects(t.x, t.y, t.z, t.def, farmIdOwner)
        if #list > 0 and total > EPS then
            self.addContext = { list=list, total=total, count=#list, farmId=farmIdOwner, fu=t.fu, ft=t.ft, srcId=(t.object and NetworkUtil.getObjectId(t.object) or 0), unitText=getUnitTextFor(t.object, t.fu) }
            if TakeObjectMain.actionEventIdAdd ~= nil and g_inputBinding ~= nil then
                local label = string.format("%s [+%.0f %s | %dx]", g_i18n:getText("input_ADD_OBJECTS"), total, self.addContext.unitText or "L", #list)
                g_inputBinding:setActionEventText(TakeObjectMain.actionEventIdAdd, label)
                if g_inputBinding.setActionEventTextVisibility ~= nil then g_inputBinding:setActionEventTextVisibility(TakeObjectMain.actionEventIdAdd, true) end
            end
        end
    end
end

function TakeObjectMain:onActionTakeObject(_, keyStatus)
    if keyStatus ~= 1 then return end
    local t = self.currentTarget; if t == nil then return end

    local wish = (t.def and t.def.object and t.def.object.liters) or DEFAULT_OBJ_LITERS
    local liters = computeFirstObjectLiters(t.lvl or 0, wish, (t.def and t.def.object and t.def.object.allowRemainder) == true)
    if liters <= 0 then return end

    local nx, ny, nz = t.nx or 0, t.ny or 0, t.nz or 1
    local pxOut = (t.x or 0) + nx * 0.25
    local pyOut = (t.y or 0)
    local pzOut = (t.z or 0) + nz * 0.25

    local farmId = (g_localPlayer and g_localPlayer.farmId) or (g_currentMission and g_currentMission.getFarmId and g_currentMission:getFarmId()) or 1
    local srcId  = t.object and NetworkUtil.getObjectId(t.object) or 0
    local ft     = t.ft or (t.def and t.def.object and t.def.object.fillTypeIndex) or nil
    local fu     = t.fu or 1

    if g_client ~= nil then
        g_client:getServerConnection():sendEvent(SpawnObjectEvent.new(pxOut,pyOut,pzOut, ft, liters, srcId, fu, farmId))
    elseif g_server ~= nil then
        local ev = SpawnObjectEvent.new(pxOut,pyOut,pzOut, ft, liters, srcId, fu, farmId)
        ev:run(g_server)
    end
end

function TakeObjectMain:onActionAddObjects(_, keyStatus)
    if keyStatus ~= 1 then return end
    local ctx = self.addContext; if ctx == nil or (ctx.count or 0) <= 0 then return end
    local ids = {}; for _, e in ipairs(ctx.list) do ids[#ids+1] = e.id end
    if g_client ~= nil then
        g_client:getServerConnection():sendEvent(ConsumeObjectsEvent.new(ctx.srcId, ctx.fu, ctx.ft, ids))
    elseif g_server ~= nil then
        local ev = ConsumeObjectsEvent.new(ctx.srcId, ctx.fu, ctx.ft, ids)
        ev:run(g_server)
    end
end

-- Server-Seite: Abzug
function TakeObjectMain:deduct(obj, fu, ft, liters, farmId)
    local removed = 0; farmId = farmId or 1
    if obj == nil or liters == nil or liters <= 0 then return 0 end
    if obj.addFillUnitFillLevel ~= nil then
        local ok, add = pcall(function() return obj:addFillUnitFillLevel(farmId, fu, -liters, ft, ToolType.UNDEFINED, nil) or 0 end)
        if ok then removed = math.max(0, -add) end
    end
    if removed <= EPS and obj.getFillUnitFillLevel ~= nil then
        local cur = obj:getFillUnitFillLevel(fu) or 0
        local newLevel = math.max(0, cur - liters)
        if obj.setFillUnitFillLevel ~= nil then pcall(function() obj:setFillUnitFillLevel(fu, newLevel, ft, false) end) end
        if obj.updateFillUnitFillLevel ~= nil then pcall(function() obj:updateFillUnitFillLevel(fu, newLevel) end) end
        if obj.raiseDirtyFlags ~= nil then
            if obj.fillUnitDirtyFlag ~= nil then pcall(function() obj:raiseDirtyFlags(obj.fillUnitDirtyFlag) end) end
            if obj.dirtyFlag ~= nil then pcall(function() obj:raiseDirtyFlags(obj.dirtyFlag) end) end
        end
        removed = math.min(liters, cur)
    end
    local lvlAfter = (obj.getFillUnitFillLevel and obj:getFillUnitFillLevel(fu)) or 0
    if lvlAfter <= EPS and obj.delete ~= nil then pcall(function() obj:delete() end) end
    return removed
end

-- Konsolenbefehle
function TakeObjectMain:consoleCommand_tsoList()
    local n=0
    for id, def in pairs(TakeSingleObjects.registry) do
        local ft = def.object and (def.object.fillType or def.object.fillTypeIndex)
        print(string.format("[TSO] id=%s api=%s ends=%s objXml=%s L=%d ft=%s consumable=%s configurable=%s", tostring(id), tostring(def.apiVersion), tostring(def.__ends), tostring(def.object and def.object.xml), def.object and def.object.liters or 0, tostring(ft), tostring(def.object and def.object.isConsumable), tostring(def.object and def.object.configurable)))
        n=n+1
    end
    print(string.format("[TSO] mappings=%d", n))
    return n
end

function TakeObjectMain:consoleCommand_tsoReload()
    TakeSingleObjects._mappingsLoaded = false
    ensureMappingsLoaded()
    return self:consoleCommand_tsoList()
end

addConsoleCommand("tsoList",   "Lists all TakeSingleObjects mappings", "consoleCommand_tsoList",  TakeObjectMain)
addConsoleCommand("tsoReload", "Reloads tso_mappings.xml from all mods", "consoleCommand_tsoReload", TakeObjectMain)

-- Action-Registrierung
local function registerAction(name, handler, idField)
    if g_inputBinding == nil then return end
    local a = InputAction and InputAction[name]
    if a == nil then wlog("Action %s nicht gefunden (modDesc.xml prüfen)", tostring(name)); return end
    local _, id = g_inputBinding:registerActionEvent(a, TakeObjectMain, handler, false, true, false, true, nil, true)
    TakeObjectMain[idField] = id
    if id ~= nil and g_inputBinding.setActionEventTextPriority then g_inputBinding:setActionEventTextPriority(id, GS_PRIO_VERY_HIGH) end
    if id ~= nil and g_inputBinding.setActionEventTextVisibility then g_inputBinding:setActionEventTextVisibility(id, false) end
end

function TakeObjectMain:registerActions()
    if TakeObjectMain.actionEventIdTake == nil then
        registerAction("TAKE_OBJECT", TakeObjectMain.onActionTakeObject, "actionEventIdTake")
    end
    if TakeObjectMain.actionEventIdAdd == nil then
        registerAction("ADD_OBJECTS", TakeObjectMain.onActionAddObjects, "actionEventIdAdd")
    end
end

Player.registerActionEvents = Utils.appendedFunction(Player.registerActionEvents, function(self, ...) TakeObjectMain:registerActions() end)
PlayerInputComponent.registerActionEvents = Utils.appendedFunction(PlayerInputComponent.registerActionEvents, function(self, ...) TakeObjectMain:registerActions() end)
PlayerInputComponent.registerGlobalPlayerActionEvents = Utils.appendedFunction(PlayerInputComponent.registerGlobalPlayerActionEvents, function(self, ...) TakeObjectMain:registerActions() end)

addModEventListener(TakeObjectMain)
