-- File: scripts/events/SpawnObjectEvent.lua
--[[ 
Mod: Take Single Objects

Author: Lism
Date: 2025-09-13
Version: 1.0.0.0

Changelog:
    v 1.0.0.0 @2025-09-13 - Initial release
--]]

SpawnObjectEvent = {}
local SpawnObjectEvent_mt = Class(SpawnObjectEvent, Event)
InitEventClass(SpawnObjectEvent, "SpawnObjectEvent")

local EPS = 0.001
local DBG = false

local function dlog(fmt, ...) if DBG then Logging.info("[TakeSingleObjects] " .. string.format(fmt, ...)) end end

local MOD_DIR = g_currentModDirectory
if (MOD_DIR == nil or MOD_DIR == "") and g_modManager ~= nil and g_currentModName ~= nil then
    local m = g_modManager:getModByName(g_currentModName)
    if m ~= nil and m.modDir ~= nil then MOD_DIR = m.modDir end
end

local function normPath(p) if p == nil then return "" end p = string.gsub(p, "\\", "/"); return string.lower(p) end
local function objectConfigPath(obj) local cfg = obj and (obj.configFileName or obj.configFile) or ""; return normPath(cfg) end
local function endsWith(str, suffix) if str == nil or suffix == nil then return false end local ls, le = #str, #suffix if le > ls then return false end return string.sub(str, ls-le+1, ls) == suffix end

local function matchDefForObject(obj)
    if TakeSingleObjects == nil or TakeSingleObjects.registry == nil then return nil end
    local cfg = objectConfigPath(obj)
    for _, def in pairs(TakeSingleObjects.registry) do
        local e = def.__ends or (def.match and def.match.endsWith)
        if e ~= nil and e ~= "" and endsWith(cfg, e) then return def end
    end
    return nil
end

local function getConsumableTypeByFU(obj, fu)
    local spec = obj and obj.spec_consumable
    if spec and spec.types then
        for i, t in ipairs(spec.types) do
            if t.fillUnitIndex == fu then return t, i end
        end
    end
    return nil, nil
end

local function pickLowestOccupiedSlotIndex(typeObj)
    if typeObj == nil or typeObj.storageSlots == nil then return nil end
    local idx
    for i=1, #typeObj.storageSlots do
        local s = typeObj.storageSlots[i]
        if (s.consumableVariationIndex or 0) > 0 then idx = i break end
    end
    return idx
end

local function findTypeIndexForVariation(vehicle, globalVarIndex)
    if vehicle == nil or vehicle.spec_consumable == nil or globalVarIndex == nil then return 1 end
    for ti, t in ipairs(vehicle.spec_consumable.types or {}) do
        for _, s in ipairs(t.storageSlots or {}) do
            if s and s.consumableVariationIndex == globalVarIndex then return ti end
        end
    end
    return 1
end

local function readbackCurrentVar(v, typeIndex)
    if v == nil or v.spec_consumable == nil then return nil end
    local t = v.spec_consumable.types and v.spec_consumable.types[typeIndex]
    local s = t and t.storageSlots and t.storageSlots[1]
    return s and s.consumableVariationIndex or 0
end

local function refreshPalletAfterChange(pallet)
    if pallet == nil then return end
    if pallet.setPalletTensionBeltNodesDirty ~= nil then pcall(function() pallet:setPalletTensionBeltNodesDirty() end) end
    if pallet.updatePalletStraps ~= nil then pcall(function() pallet:updatePalletStraps() end) end
end

function SpawnObjectEvent.emptyNew() return Event.new(SpawnObjectEvent_mt) end

function SpawnObjectEvent.new(x, y, z, fillTypeIndex, liters, srcNetId, fu, farmId)
    local self = SpawnObjectEvent.emptyNew()
    self.x, self.y, self.z = x, y, z
    self.fillTypeIndex = fillTypeIndex or 0
    self.liters = liters or 0
    self.srcNetId = srcNetId or 0
    self.fu = fu or 1
    self.farmId = farmId or 0 -- wird serverseitig überschrieben (Owner der Quelle)
    return self
end

function SpawnObjectEvent:readStream(streamId, connection)
    self.x = streamReadFloat32(streamId)
    self.y = streamReadFloat32(streamId)
    self.z = streamReadFloat32(streamId)
    self.fillTypeIndex = streamReadUIntN(streamId, FillTypeManager.SEND_NUM_BITS)
    self.liters = streamReadFloat32(streamId)
    self.srcNetId = NetworkUtil.readNodeObjectId(streamId)
    self.fu = streamReadUInt8(streamId)
    if FarmManager ~= nil and FarmManager.SEND_NUM_BITS ~= nil then self.farmId = streamReadUIntN(streamId, FarmManager.SEND_NUM_BITS) else self.farmId = streamReadUInt8(streamId) end
    self:run(connection)
end

function SpawnObjectEvent:writeStream(streamId, connection)
    streamWriteFloat32(streamId, self.x)
    streamWriteFloat32(streamId, self.y)
    streamWriteFloat32(streamId, self.z)
    streamWriteUIntN(streamId, self.fillTypeIndex, FillTypeManager.SEND_NUM_BITS)
    streamWriteFloat32(streamId, self.liters)
    NetworkUtil.writeNodeObjectId(streamId, self.srcNetId)
    streamWriteUInt8(streamId, self.fu or 1)
    if FarmManager ~= nil and FarmManager.SEND_NUM_BITS ~= nil then streamWriteUIntN(streamId, self.farmId or 0, FarmManager.SEND_NUM_BITS) else streamWriteUInt8(streamId, self.farmId or 0) end
end

function SpawnObjectEvent:run(connection)
    if g_currentMission == nil or not g_currentMission:getIsServer() then dlog("SpawnObjectEvent:run on client -> ignore"); return end

    local x, y, z = self.x, self.y, self.z
    local ft, liters = self.fillTypeIndex or 0, self.liters or 0
    local fu = self.fu or 1

    local srcObj = NetworkUtil.getObject(self.srcNetId)
    if srcObj == nil then return end

    local farmId = 1
    if srcObj.getOwnerFarmId ~= nil then
        local owner = srcObj:getOwnerFarmId() or 0
        if owner ~= 0 then farmId = owner end
    end

    local def = matchDefForObject(srcObj)
    if def == nil then dlog("No mapping for source -> skip"); return end

    local removed = 0
    local consumableVarIndex = nil -- globaler Index

    if def.object.isConsumable and srcObj.spec_consumable ~= nil then
        local typeObj, typeIndex = getConsumableTypeByFU(srcObj, fu)
        if typeObj == nil then return end
        local slotIndex = pickLowestOccupiedSlotIndex(typeObj)
        if slotIndex == nil then return end -- leer

        local slot = typeObj.storageSlots[slotIndex]
        consumableVarIndex = slot and slot.consumableVariationIndex or 0
        dlog("Entnahme Slot=%d var(global)=%s", slotIndex, tostring(consumableVarIndex))

        -- Slot leeren & Spec updaten
        local ok1 = pcall(function() return srcObj:setConsumableSlotVariationIndex(typeIndex or 1, slotIndex, 0) end)
        if not ok1 then return end
        if srcObj.updateConsumable ~= nil then pcall(function() srcObj:updateConsumable(typeObj.typeName, 0, false, true) end) end
        refreshPalletAfterChange(srcObj)

        -- Füllstand reduzieren (1 Stück)
        local ok2, _ = pcall(function() return srcObj:addFillUnitFillLevel(farmId, fu, -1, ft, ToolType.UNDEFINED, nil) end)
        if not ok2 then return end
        removed = 1
    else
        local wanted = def.object.liters or 1
        if liters > 0 then wanted = math.min(wanted, liters) end
        if TakeObjectMain ~= nil and TakeObjectMain.deduct ~= nil then removed = TakeObjectMain:deduct(srcObj, fu, ft, wanted, farmId) or 0 end
    end

    if (removed or 0) <= EPS then dlog("No liters removed -> skip spawn"); return end

    local xmlPath = def.object.xml or "bag/bag.xml"
    SpawnObjectEvent.spawnObject(xmlPath, x, y, z, ft, removed, farmId, consumableVarIndex)

    if g_server ~= nil and connection ~= nil then g_server:broadcastEvent(self, false, connection) end
end

function SpawnObjectEvent.spawnObject(xmlPath, x, y, z, fillTypeIndex, liters, farmId, globalVarIndex)
    if g_currentMission == nil then return end

    local full = xmlPath
    if not string.find(xmlPath, ":/", 1, true) and string.sub(xmlPath, 1, 1) ~= "/" and string.sub(xmlPath, 1, 1) ~= "$" then full = Utils.getFilename(xmlPath, MOD_DIR) end

    farmId = farmId or 1
    local terrainY = getTerrainHeightAtWorldPos(g_terrainNode, x, y, z)
    local spawnY = math.max(y, terrainY + 0.05)

    local function onVehicleLoaded(_, vehicles, vehicleLoadState, arguments)
        local vehicle = vehicles and vehicles[1] or nil
        if vehicle == nil then Logging.warning("[TakeSingleObjects] VehicleLoadingData: no vehicle for %s", tostring(full)); return end
        pcall(function() vehicle:setOwnerFarmId(farmId, true) end)

        -- 1) Variation setzen (globaler Index vom Quell‑Slot)
        local ti = findTypeIndexForVariation(vehicle, globalVarIndex)
        if globalVarIndex ~= nil and globalVarIndex > 0 and vehicle.setConsumableSlotVariationIndex ~= nil then
            pcall(function() vehicle:setConsumableSlotVariationIndex(ti, 1, 0) end)
            if vehicle.updateConsumable ~= nil then pcall(function() vehicle:updateConsumable(vehicle.spec_consumable.types[ti].typeName, 0, false, true) end) end
            pcall(function() vehicle:setConsumableSlotVariationIndex(ti, 1, globalVarIndex) end)
            if vehicle.updateConsumable ~= nil then pcall(function() vehicle:updateConsumable(vehicle.spec_consumable.types[ti].typeName, 0, false, true) end) end
        end
        dlog("Spawn set var(global)=%s → typeIdx=%d (preFill readback=%s)", tostring(globalVarIndex), ti, tostring(readbackCurrentVar(vehicle, ti)))

        -- 2) FillUnits befüllen
        local remaining = liters or 0
        if remaining > 0 and vehicle.getFillUnits ~= nil then
            pcall(function() vehicle:emptyAllFillUnits(true) end)
            for _, fu in ipairs(vehicle:getFillUnits() or {}) do
                local fuIndex = fu.fillUnitIndex or fu.index or 1
                if vehicle.getFillUnitSupportsFillType ~= nil and vehicle:getFillUnitSupportsFillType(fuIndex, fillTypeIndex) then
                    local ok2, _ = pcall(function() return vehicle:addFillUnitFillLevel(farmId, fuIndex, remaining, fillTypeIndex, ToolType.UNDEFINED, nil) end)
                    if ok2 then remaining = 0 break end
                end
            end
            if remaining > 0 then
                for _, fu in ipairs(vehicle:getFillUnits() or {}) do
                    local fuIndex = fu.fillUnitIndex or fu.index or 1
                    if vehicle.getFillUnitSupportsFillType ~= nil and vehicle:getFillUnitSupportsFillType(fuIndex, fillTypeIndex) then
                        pcall(function() vehicle:setFillUnitFillLevel(fuIndex, liters, fillTypeIndex, false) end)
                        remaining = 0
                        break
                    end
                end
            end
        end

        -- 3) Variation zur Sicherheit erneut setzen
        if globalVarIndex ~= nil and globalVarIndex > 0 and vehicle.setConsumableSlotVariationIndex ~= nil then
            local cur = readbackCurrentVar(vehicle, ti)
            if cur ~= globalVarIndex then
                pcall(function() vehicle:setConsumableSlotVariationIndex(ti, 1, globalVarIndex) end)
                if vehicle.updateConsumable ~= nil then pcall(function() vehicle:updateConsumable(vehicle.spec_consumable.types[ti].typeName, 0, false, true) end) end
            end
        end

        dlog("Spawned item %s at %.2f/%.2f/%.2f (ft=%s, L=%.1f)", tostring(full), x, spawnY, z,
            g_fillTypeManager and g_fillTypeManager:getFillTypeNameByIndex(fillTypeIndex) or tostring(fillTypeIndex), liters or 0)
    end

    local data = VehicleLoadingData.new()
    data:setFilename(full)
    data:setPosition(x, spawnY, z)
    data:setRotation(0, 0, 0)
    data:setPropertyState(VehiclePropertyState.OWNED)
    data:setOwnerFarmId(farmId)
    data:load(onVehicleLoaded)
end
