--[[
    ToolBoxPickupFixes

	Fixes the dynamic mount attacher's behavior to enable pickup of dynamic mounted objects by hand

	@author: 		BayernGamers
	@date: 			25.04.2025
	@version:		1.0

	History:		v1.0 @25.04.2025 - initial implementation in FS 25
                    ------------------------------------------------------------------------------------------------------
	
	License:        Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage: 
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.
						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]
source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))

local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.HIGH, "ToolBoxPickupFixes.lua")

ToolBoxPickupFixes = {}

function ToolBoxPickupFixes:getCanPickUpNode(superFunc, nodeId)
	local result = superFunc(self, nodeId)
	local nodeObject = g_currentMission:getNodeObject(nodeId)

    if nodeObject ~= nil then
        if nodeObject.dynamicMountObject ~= nil then
			result = true
		end

		local spec = self.spec_hands
		if not spec.hasSuperStrength and (nodeObject.getCanBePickedUp ~= nil and not nodeObject:getCanBePickedUp(self:getCarryingPlayer())) then
            result = false
        end
	end

    return result
end

HandToolHands.getCanPickUpNode = Utils.overwrittenFunction(HandToolHands.getCanPickUpNode, ToolBoxPickupFixes.getCanPickUpNode)

function ToolBoxPickupFixes:pickUpTarget(superFunc, target, skipEvent)
    
	if self.isServer then
		local nodeObject = g_currentMission:getNodeObject(target.node)

		if nodeObject ~= nil and nodeObject.unmountDynamic ~= nil then
			nodeObject:unmountDynamic()
		end
	end

	return superFunc(self, target, skipEvent)
end

HandToolHands.pickUpTarget = Utils.overwrittenFunction(HandToolHands.pickUpTarget, ToolBoxPickupFixes.pickUpTarget)

function ToolBoxPickupFixes:addDynamicMountedObject(superFunc, object)
    local spec = self.spec_dynamicMountAttacher
    if spec.dynamicMountedObjects[object] == nil then
        spec.dynamicMountedObjects[object] = object
        local isLocked = false
        
        if object.getMountableLockPositions ~= nil then
            local lockPositions = object:getMountableLockPositions()
            for i = 1, #lockPositions do
                local lockPosition = lockPositions[i]
                if string.endsWith(self.configFileName, lockPosition.xmlFilename) then
                    local jointNode = I3DUtil.indexToObject(self.components, lockPosition.jointNode, self.i3dMappings)
                    if jointNode ~= nil then
                        local x, y, z = localToWorld(jointNode, lockPosition.transOffset[1], lockPosition.transOffset[2], lockPosition.transOffset[3])
                        local rotX, rotY, rotZ = localRotationToWorld(jointNode, lockPosition.rotOffset[1], lockPosition.rotOffset[2], lockPosition.rotOffset[3])
                        self:lockDynamicMountedObject(object, x, y, z, rotX, rotY, rotZ)
                        isLocked = true
                        break
                    end
                end
            end
        end
        
        if not isLocked and object:isa(Vehicle) then
            for i = 1, #spec.lockPositions do
                local lockPosition = spec.lockPositions[i]
                if string.endsWith(object.configFileName, lockPosition.xmlFilename) then
                    local x, y, z = getWorldTranslation(lockPosition.jointNode)
                    local rotX, rotY, rotZ = getWorldRotation(lockPosition.jointNode)
                    self:lockDynamicMountedObject(object, x, y, z, rotX, rotY, rotZ)
                    ObjectChangeUtil.setObjectChanges(lockPosition.objectChanges, true, self, self.setMovingToolDirty)
                    break
                end
            end
        end
        
        if spec.transferMass and (object.setReducedComponentMass ~= nil and object:getAllowComponentMassReduction()) then
            object:setReducedComponentMass(true)
            self:setMassDirty()
        end
        
        self:setDynamicMountAnimationState(true)
        self:raiseDirtyFlags(spec.dynamicMountedObjectsDirtyFlag)
    end
end

DynamicMountAttacher.addDynamicMountedObject = Utils.overwrittenFunction(DynamicMountAttacher.addDynamicMountedObject, ToolBoxPickupFixes.addDynamicMountedObject)